/*
 * class: JInstall
 * Copyright (C) 2002  Marcello Balduccini
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * marcello.balduccini@ttu.edu
 *
 */

package jinstall;

import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;
import java.util.*;

import javax.swing.*;

abstract public class JInstall extends JFrame implements ActionListener, WindowListener, Runnable
{
	private JTextField tx1;
	private JTextArea tx2;
	private JButton bt,br,btquit;
	private JProgressBar pb;

	private boolean local_download;

	protected File stdout,stderr,temp_dir,script;
	protected String stdoutfile, stderrfile,TEMP_DIR,SCRIPT;

	protected final static String VERSION="1.2";
	protected final static String DATE="08/08/02";

	protected String NAME,
			 PREFIX,
			 DEFAULT_PATH="/usr/local/",
			 BIN_DIR="bin/";

	private boolean _abortTasks;

	/* task variables */
	protected File dest;

	protected JInstall(String name,String prefix,boolean loc_dw)
	{
		super("JInstaller v"+VERSION+" for "+name);

		NAME=name;
		PREFIX=prefix;

		local_download=loc_dw;

		SCRIPT="";
		try
		{	temp_dir=File.createTempFile(PREFIX+"_dir",null);
			temp_dir.delete();
			temp_dir.mkdir();
			TEMP_DIR=temp_dir.getAbsolutePath();

			stdout=File.createTempFile(PREFIX+"_out",null);
			stdout.deleteOnExit();
			stdoutfile=stdout.getAbsolutePath();
			stderr=File.createTempFile(PREFIX+"_err",null);
			stderr.deleteOnExit();
			stderrfile=stderr.getAbsolutePath();

			script=File.createTempFile(PREFIX+"_script",".sh");
			script.deleteOnExit();
			writeScript(script);
			SCRIPT=script.getAbsolutePath();
		}
		catch (Exception e)
		{	System.out.println("Unable to create temporary files. Aborting.");
			deleteRecursive(temp_dir);
			System.exit(1);
		}

		Container ct=getContentPane();
		JPanel p;


		ct.setLayout(new BorderLayout());
		ct.add("North",p=new JPanel(new FlowLayout()));

		p.add(new JLabel("Install path:"));
		tx1=new JTextField(DEFAULT_PATH,30);
		p.add(tx1);

		bt=new JButton("Install");
		p.add(bt);
		bt.addActionListener(this);

		btquit=new JButton("Quit");
		p.add(btquit);
		btquit.addActionListener(this);

		tx2=new JTextArea();
		tx2.setRows(20);
		ct.add("Center",new JScrollPane(tx2));

		pb=new JProgressBar(/*0,N_TASKS*/);
		pb.setBorderPainted(true);
		pb.setStringPainted(true);
		ct.add("South",pb);
		pb.setVisible(false);

		addWindowListener(this);
		pack();

		clearLog();

		setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);

		setVisible(true);
	}

	protected String getDestDir()
	{	return(tx1.getText());
	}

	private void writeScript(File f) throws Exception, IOException
	{
		PrintWriter pw=new PrintWriter(new FileWriter(f));

		pw.println("#! /bin/sh");
		pw.println("#");
		pw.println("# Usage: "+f.getAbsolutePath()+" <dir> <stdout file> <stderr file> <path addition> <cmdline>");
		pw.println("#");
		pw.println("");
		pw.println("DIR=`pwd`");
		pw.println("");
		pw.println("newdir=$1");
		pw.println("shift");
		pw.println("stdout=$1");
		pw.println("shift");
		pw.println("stderr=$1");
		pw.println("shift");
		pw.println("PATH=$1:$PATH");
		pw.println("export PATH");
		pw.println("shift");
		pw.println("");
		pw.println("cd $newdir");
		pw.println("exec $* >$stdout 2>$stderr");

		pw.close();

		exec("chmod a+rx "+f.getAbsolutePath(),".",false,false);
	}

	protected void deleteRecursive(File f,boolean rmThis)
	{	
		if (f.isDirectory())
		{	File l[];
			int i;

			l=f.listFiles();
			for(i=0;i<l.length;i++)
				deleteRecursive(l[i],true);
		}

		if (rmThis)
			f.delete();
	}

	protected void deleteRecursive(File f)
	{	deleteRecursive(f,true);
	}

	protected void clearLog()
	{	tx2.setText("JInstaller v"+VERSION+" for "+NAME+"\n");
		tx2.append( "==============\n");
		tx2.append( "  Author: Marcello Balduccini (marcello.balduccini@ttu.edu)\n");
		tx2.append( "  Date: "+DATE+"\n");
		tx2.append( "==============\n\n");
		tx2.append( "Please select the path where you want the files to be installed\n");
		tx2.append( "and click Install.\n\n");
		tx2.append( "IMPORTANT: if you wish to install to the default path /usr/local,\n");
		tx2.append( "make sure to run this installer as 'root'\n\n");
	}

	protected void println(String s)
	{	tx2.append(s+"\n");
//		System.out.println(s);

		tx2.setCaretPosition(tx2.getText().length());
	}

	private void taskDone()
	{	pb.setValue(pb.getValue()+1);
	}

	private String readErrorFile(String fname)
	{	BufferedReader t;
		String line,str;

		str="";
		try
		{
			t=new BufferedReader(new FileReader(fname));
			while ((line=t.readLine())!=null)
				str=str+line+"\n";
			t.close();
		}
		catch(IOException ix)
		{	/* just ignore, we are only reading an error file */
		}

		return(str);
	}

	protected boolean exec(String cmd,String dir,boolean verbose,boolean use_script) throws Exception
	{	Process p;
		String str,title,prefix;
		int n;
		boolean ret;

		try
		{	setCursor(new Cursor(Cursor.WAIT_CURSOR));

			if (use_script)
			{	if (dest!=null)
					prefix=dest.getAbsolutePath()+"/"+BIN_DIR;
				else
					prefix="/bin";	/* dummy path addition */

				p=Runtime.getRuntime().exec(SCRIPT+" "+dir+" "+stdoutfile+" "+stderrfile+" "+prefix+" "+cmd);
			}
			else
				p=Runtime.getRuntime().exec(cmd);
			p.waitFor();
		}
		catch(Exception e)
		{	setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
			println("Error while trying to execute \""+cmd+"\".\n"+e);
			//ResultDialog.popup(this,"Error","Error while trying to execute \""+cmd+"\".\n"+e,"");
			println("Error while trying to execute \""+cmd+"\".\n"+e);
			throw(e);
//			return(false);
		}
		setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
		if ((n=p.exitValue())!=0)
		{	str="Error number "+n+" while executing \""+cmd+"\"";
			title="Error";
			ret=true;
		}
		else
		{	str="Successful execution of \""+cmd+"\"";
			title="Success";
			ret=false;
		}
		if ((ret) || (verbose))
			println(str+"\n\nStdout:\n"+"-------\n"+readErrorFile(stdoutfile)+"\nStderr:\n"+"-------\n"+readErrorFile(stderrfile));
//			ResultDialog.popup(this,title,str+"\n\nStdout:\n"+"-------\n"+readErrorFile(stdoutfile),"\nStderr:\n"+"-------\n"+readErrorFile(stderrfile));

		if (ret)
			throw(new Exception());

		return(ret);
	}

	protected boolean exec(String cmd,String dir,boolean verbose) throws Exception
	{	return(exec(cmd,dir,verbose,true));
	}

	protected boolean exec(String cmd) throws Exception
	{	return(exec(cmd,".",false));
	}

	protected boolean exec(String cmd,String dir) throws Exception
	{	return(exec(cmd,dir,false));
	}

	protected boolean exec_verbose(String cmd) throws Exception
	{	return(exec(cmd,".",true));
	}

	protected boolean exec_verbose(String cmd,String dir) throws Exception
	{	return(exec(cmd,dir,true));
	}
/*
	public void run_ping()
	{	exec_verbose("ping -o "+host);
	}
*/

	protected void getFile(String web,String path,String fname,String destdir) throws MalformedURLException, IOException, Exception
	{	try
		{	URL u = new URL(web+"/"+path+"/"+fname);


			if (local_download)
			{	exec("cp DW/"+fname+" "+destdir);
				return;
			}

			InputStream in = u.openStream();
			FileOutputStream out=new FileOutputStream(destdir+"/"+fname);
			byte b[]=new byte[1024];
			int ret;

			ret=in.read(b);
			while(ret!=-1)
			{	out.write(b,0,ret);
				ret=in.read(b);
			}

			in.close();
			out.close();
		}
		catch (MalformedURLException e)
		{	println(e.toString());
			throw(e);
		}
		catch (IOException e)
		{	println(e.toString());
			throw(e);
		}
		catch (Exception e)
		{	println(e.toString());
			throw(e);
		}
	}

	protected void setEnabledAll(boolean s)
	{	bt.setEnabled(s);
		btquit.setEnabled(s);
		tx1.setEnabled(s);
	}

	protected String getPath(String f,String s)
	{	File f2;

		f2=new File(f,s);
		return(f2.getPath());
	}

	protected void abortTasks()
	{	_abortTasks=true;
	}

	protected abstract void fillTasks(Vector t);

	public void run()
	{	Vector t;
		int i;

		t=new Vector();
		fillTasks(t);

		pb.setMaximum(t.size());
		pb.setValue(0);
		pb.setVisible(true);

		_abortTasks=false;
		for(i=0;i<t.size() && !_abortTasks;i++)
		{	((InstallTask)(t.elementAt(i))).run();
			taskDone();
		}
		setEnabledAll(true);
	}

	private void terminate()
	{	deleteRecursive(temp_dir);
		System.exit(0);
	}

	protected String getConfirmMessage()
	{
		return("The installation process is going to begin.\n\n"+
		       "You most likely have to run this installer as root\n"+
		       "in order to install the software out of your home directory.\n\n"+
		       "Do you want to proceed ?\n");
	}

	public void actionPerformed(ActionEvent e)
	{	File dest;

		if (e.getSource()==bt)
		{	int ret;

			ret=JOptionPane.showConfirmDialog(this,getConfirmMessage(),
							       "Confirmation", JOptionPane.YES_NO_OPTION);
			if (ret==JOptionPane.YES_OPTION)
			{	setEnabledAll(false);
				new Thread(this).start();
			}
		}
		else
		if (e.getSource()==btquit)
		{	terminate();
		}
	}

	public void windowActivated(WindowEvent e)
	{}

	public void windowClosed(WindowEvent e)
	{	terminate();
	}

	public void windowClosing(WindowEvent e)
	{}

	public void windowDeactivated(WindowEvent e)
	{}

	public void windowDeiconified(WindowEvent e)
	{}

	public void windowIconified(WindowEvent e)
	{}

	public void windowOpened(WindowEvent e)
	{}

	protected static boolean parseArgs(String args[])
	{	if ((args.length==1) && (args[0].equals("-local")))
			return(true);
		return(false);
	}
}
