// Copyright (c) 2008, Roland Kaminski
//
// This file is part of GrinGo.
//
// GrinGo is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// GrinGo is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with GrinGo.  If not, see <http://www.gnu.org/licenses/>.

#include "statementdependencygraph.h"
#include "literaldependencygraph.h"
#include "statementchecker.h"
#include "tatom_predicateliteral.h"
#include "assignmentliteral.h"
#include "relationliteral.h"
#include "grounder.h"
#include "variable.h"
#include "term.h"
#include "domain.h"
#include "rangeterm.h"
#include "rangeliteral.h"
#include "indexeddomain.h"
#include "dlvgrounder.h"
#include "value.h"
#include "multipleargsterm.h"
#include "output.h"
#include "evaluator.h"
#include "bindersplitter.h"
#include "gringoexception.h"

#define ASPF_DEBUG_LIGHT

using namespace NS_GRINGO;

//#define OLDSTYLE_connectorString
//#define NEWSTYLE_CLEARNEQ_connectorString
//#define OLDSTYLE_isFact
//#define OLDSTYLE_match
//#define OLDSTYLE_convert

TermVector *TAtomPredicateLiteral::createTermVector(Term *a,Term *b)
{	TermVector *tv=new TermVector();
	tv->push_back(a);
	tv->push_back(b);

	return(tv);
}

std::string TAtomPredicateLiteral::connectorString(RelationType type)
{	std::string s;

#ifdef OLDSTYLE_connectorString
	return("");
#else
	switch(type)
	{
		case EQ:
			s = "#=";
			break;
#ifdef NEWSTYLE_CLEARNEQ_connectorString
		case NE:
		case GT:
		case GE:
		case LE:
		case LT:
			s = "";
			break;
#else
		case NE:
			s = "#!=";
			break;
		case GT:
			s = "#>";
			break;
		case GE:
			s = "#>=";
			break;
		case LE:
			s = "#<=";
			break;
		case LT:
			s = "#<";
			break;
#endif
		default:
			assert(false);

	}
	return(s);
#endif
}

TAtomPredicateLiteral::TAtomPredicateLiteral(Grounder *g, RelationType type, Term *a, Term *b) : PredicateLiteral(g,g->createString(connectorString(type)/*""*/),createTermVector(a,b)), type_(type), g_(g), headTAtom_(false)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "TAtomPredicateLiteral::TAtomPredicateLiteral(" << connectorString(type) << ") " << this << std::endl;
#endif
}

TAtomPredicateLiteral::TAtomPredicateLiteral(const TAtomPredicateLiteral &p) : PredicateLiteral(p), /*constraint_(p.constraint_), value_(p.value_),*/ type_(p.type_) /*, a_(p.a_->clone()), b_(p.b_->clone()) */, g_(p.g_), headTAtom_(p.headTAtom_)
{	//a_=(*variables_)[0];
	//b_=(*variables_)[1];
}

void TAtomPredicateLiteral::createNode(StatementChecker *dg, bool head, bool delayed)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "TAtomPredicateLiteral::createNode3() " << this << std::endl;
#endif
	if(type_!=EQ)
	{	VarSet vars, empty;
		getVars(vars);
		dg->createNode(vars, empty);
	}
	else
		PredicateLiteral::createNode(dg,head,delayed);
}

void TAtomPredicateLiteral::print(const GlobalStorage *g, std::ostream &out) const
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "TAtomPredicateLiteral::print()" << std::endl;
#endif
	if(getNeg())
		out << "not ";
	out << pp(g,(*variables_)[0]) << " ";
	out << connectorString(type_) << " ";
	out << pp(g,(*variables_)[1]);
}

/*
const ValueVector &TAtomPredicateLiteral::getValues()
{
	return values_;
}
*/

/*
bool TAtomPredicateLiteral::solved()
{
std::cerr << "tatom solved()" << std::endl;
	return predNode_->solved();
}
*/

#ifndef OLDSTYLE_isFact
bool TAtomPredicateLiteral::isFact(const ValueVector &values)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tisFact1" << std::endl;
#endif

	// Only seed t-atoms can be evaluated. All others will be evaluated by the solver.
	if (type_!=EQ || nTerms(values)!=1)
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact1a" << std::endl;
#endif
		return(false);
#ifdef ASPF_DEBUG_LIGHT
}
#endif

	// a precondition for this method is that the predicate is not false!
	if(predNode_->solved())
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact1b" << std::endl;
#endif
		return true;
#ifdef ASPF_DEBUG_LIGHT
}
#endif

	if(getNeg())
	{
		assert(!predNode_->isFact(values));
		if(predNode_->complete())
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact1c " << !predNode_->inDomain(values) << std::endl;
#endif
			return !predNode_->inDomain(values);
#ifdef ASPF_DEBUG_LIGHT
}
#endif
		else
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact1d" << std::endl;
#endif
			return false;
#ifdef ASPF_DEBUG_LIGHT
}
#endif
	}
	else
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact1c " << predNode_->isFact(values) << std::endl;
#endif
		return predNode_->isFact(values);
#ifdef ASPF_DEBUG_LIGHT
}
#endif
}
#else
///* -- WORKS, but bad performance with constraint atoms
bool TAtomPredicateLiteral::isFact(const ValueVector &values)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "isFact1" << std::endl;
#endif
	//return true;
	return false;

}
//*/
#endif

#ifndef OLDSTYLE_isFact
bool TAtomPredicateLiteral::isFact(Grounder *g)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tisFact2" << std::endl;
#endif

	for(int i = 0; i < (int)variables_->size(); i++)
		values_[i] = (*variables_)[i]->getValue(g);

	// Only seed t-atoms can be evaluated. All others will be evaluated by the solver.
	if (type_!=EQ || nTerms(values_)!=1)
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact2a" << std::endl;
#endif
		return(false);
#ifdef ASPF_DEBUG_LIGHT
}
#endif

	// a precondition for this method is that the predicate is not false!
	if(predNode_->solved())
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact2b" << std::endl;
#endif
		return true;
#ifdef ASPF_DEBUG_LIGHT
}
#endif
	for(int i = 0; i < (int)variables_->size(); i++)
		values_[i] = (*variables_)[i]->getValue(g);

	if(getNeg())
	{
		assert(!predNode_->isFact(values_));
		if(predNode_->complete())
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact2c " << !predNode_->inDomain(values_) << std::endl;
#endif
			return !predNode_->inDomain(values_);
#ifdef ASPF_DEBUG_LIGHT
}
#endif
		else
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tisFact2d" << std::endl;
#endif
			return false;
#ifdef ASPF_DEBUG_LIGHT
}
#endif
	}
	else
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "isFact2e " << predNode_->isFact(values_) << std::endl;
#endif
		return predNode_->isFact(values_);
#ifdef ASPF_DEBUG_LIGHT
}
#endif
}
#else
///* -- WORKS, but bad performance with constraint atoms
bool TAtomPredicateLiteral::isFact(Grounder *g)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "isFact2" << std::endl;
#endif
	//return true;
	return false;
}
#endif

#ifndef OLDSTYLE_match
bool TAtomPredicateLiteral::match(const ValueVector &values)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug)
{	std::cerr << "tmatch1" << std::endl;
	for(int i=0;i<values.size();i++)
	{	std::cerr << "  val(" << i << ")=" << (values[i].asString(g_)) << std::endl;
	}
}
#endif
	// Only seed t-atoms can be evaluated. All others will be evaluated by the solver.
	if (type_!=EQ || nTerms(values)!=1)
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tmatch1a" << std::endl;
#endif
		return(true);
#ifdef ASPF_DEBUG_LIGHT
}
#endif

	if(!predNode_->complete())
	{
		if(getNeg())
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tmatch1b " << !predNode_->isFact(values) << std::endl;
#endif
			return !predNode_->isFact(values);
#ifdef ASPF_DEBUG_LIGHT
}
#endif
		else
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tmatch1c" << std::endl;
#endif
			return true;
#ifdef ASPF_DEBUG_LIGHT
}
#endif
	}
	bool match = predNode_->inDomain(values);

//std::cerr << "predNode_ domain:" << std::endl;
//{
//int i=0;
//for(ValueVectorSet::iterator it = predNode_->getDomain().begin(); it != predNode_->getDomain().end(); it++)
//{	for(int j=0;j<it->size();j++)
//	{	std::cerr << "dom(" << i << "," << j << ")=" << (*it)[j].asString(g_) << std::endl;
//	}
//	i++;
//}
//}

	if(getNeg())
	{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch1d " << !(match && (predNode_->solved() || predNode_->isFact(values))) << std::endl;
#endif
		return !(match && (predNode_->solved() || predNode_->isFact(values)));
	}
	else
	{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch1e " << match << std::endl;
#endif
		return match;
	}
}
#else
///* -- WORKS, but bad performance with constraint atoms
bool TAtomPredicateLiteral::match(const ValueVector &values)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch2" << std::endl;
#endif
	return true;
}
//*/
#endif

#ifndef OLDSTYLE_match
bool TAtomPredicateLiteral::match(Grounder *g)
{
	assert(!predNode_->solved() || predNode_->complete());

#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch2" << std::endl;
#endif
	// Only seed t-atoms can be evaluated. All others will be evaluated by the solver.
	if (type_!=EQ)
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tmatch2a" << std::endl;
#endif
		return(true);
#ifdef ASPF_DEBUG_LIGHT
}
#endif

	ValueVector xvalues(variables_->size());
	for(int i = 0; i < (int)variables_->size(); i++)
		xvalues[i] = (*variables_)[i]->getValue(g);

	// Only seed t-atoms can be evaluated. All others will be evaluated by the solver.
	if (nTerms(xvalues)!=1)
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tmatch2b" << std::endl;
#endif
		return(true);
#ifdef ASPF_DEBUG_LIGHT
}
#endif

	// incomplete prediactes match everything
	// except if they contain some values that are facts!!!
	if(!predNode_->complete())
	{
		if(getNeg() && predNode_->hasFacts())
		{
			for(int i = 0; i < (int)variables_->size(); i++)
				values_[i] = (*variables_)[i]->getValue(g);
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch2c " << !predNode_->isFact(values_) << std::endl;
#endif
			return !predNode_->isFact(values_);
		}
		else
#ifdef ASPF_DEBUG_LIGHT
{
if (g_->options().debug) std::cerr << "tmatch2d" << std::endl;
#endif
			return true;
#ifdef ASPF_DEBUG_LIGHT
}
#endif
	}
	// check if current assignment is satisfied wrt the domain
	bool match = true;
	if(variables_)
	{
		for(int i = 0; i < (int)variables_->size(); i++)
			values_[i] = (*variables_)[i]->getValue(g);
	}
	match = predNode_->inDomain(values_);
	if(getNeg())
	{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch2e " << (!(match && (predNode_->solved() || predNode_->isFact(values_)))) << std::endl;
#endif
		return !(match && (predNode_->solved() || predNode_->isFact(values_)));
	}
	else
	{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch2f" << std::endl;
#endif
		return match;
	}
}
#else
///* -- WORKS, but bad performance with constraint atoms
bool TAtomPredicateLiteral::match(Grounder *g)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tmatch2" << std::endl;
#endif
	return true;
}
//*/
#endif

Clasp::CSPConstraint* TAtomPredicateLiteral::getCSPConstraintOfValue(Grounder *g,const Value& v)
{
	Clasp::CSPConstraint* c = new Clasp::CSPConstraint();
	if (v.type_ == Value::INT)
	{
		c->setInteger(v.uid_);
	}else if (v.type_ == Value::STRING | v.type_ == Value::FUNCSYMBOL)
	{
		std::string name = v.asString(g);
                c->setVariable(g->getCSPSolver()->getVariable(name));
	}
	else
	{
		assert(false);
		return 0;
	}
	return c;
}

bool TAtomPredicateLiteral::isUndefined(Term *a)
{	Value::Type t;

	t=a->getValue(g_).type_;
	return (t!=Value::INT && t!=Value::STRING && t!=Value::FUNCSYMBOL);
}

bool TAtomPredicateLiteral::isFTerm(Term *a,int idx,const ValueVector &values)
{	Value::Type t;
	int uid,arity;

	t=a->getValue(g_).type_;
	if (t!=Value::INT && t!=Value::STRING && t!=Value::FUNCSYMBOL)
		t=values[idx].type_;

	switch(t)
	{	case Value::INT:
			return(false);
		case Value::FUNCSYMBOL:
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) 
{	std::cerr << "arg is FuncSymbol: ";
	g_->getFuncSymbol(a->getValue(g_).uid_)->print(g_,std::cerr);
	std::cerr << std::endl;
}
#endif
			uid=g_->getFuncSymbol(a->getValue(g_).uid_)->getName();
			arity=g_->getFuncSymbol(a->getValue(g_).uid_)->getValues().size();
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg's functor uid=" << uid << "; arity=" << arity << std::endl;
#endif
			break;
		case Value::STRING:
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg is String" << std::endl;
#endif
			uid=a->getValue(g_).uid_;
			arity=0;
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg's string functor uid=" << uid << "; arity=" << arity << std::endl;
#endif
			break;
		default:
			std::cerr << "internal error: unknown term type " << t << std::endl;
			exit(1);
	}

	return(g_->getCSPSolver()->isFTerm(uid,arity));
}

bool TAtomPredicateLiteral::isShown(Term *a,int idx,const ValueVector &values)
{	Value::Type t;
	int uid,arity;

	t=a->getValue(g_).type_;
	if (t!=Value::INT && t!=Value::STRING && t!=Value::FUNCSYMBOL)
		t=values[idx].type_;

	switch(t)
	{	case Value::INT:
			return(true);
		case Value::FUNCSYMBOL:
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) 
{	std::cerr << "arg is FuncSymbol: ";
	g_->getFuncSymbol(a->getValue(g_).uid_)->print(g_,std::cerr);
	std::cerr << std::endl;
}
#endif
			uid=g_->getFuncSymbol(a->getValue(g_).uid_)->getName();
			arity=g_->getFuncSymbol(a->getValue(g_).uid_)->getValues().size();
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg's functor uid=" << uid << "; arity=" << arity << std::endl;
#endif
			break;
		case Value::STRING:
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg is String" << std::endl;
#endif
			uid=a->getValue(g_).uid_;
			arity=0;
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg's string functor uid=" << uid << "; arity=" << arity << std::endl;
#endif
			break;
		default:
			std::cerr << "internal error: unknown term type " << t << std::endl;
			exit(1);
	}

	return(g_->getCSPSolver()->isShown(uid,arity));
}

bool TAtomPredicateLiteral::isHidden(Term *a,int idx,const ValueVector &values)
{	Value::Type t;
	int uid,arity;

	t=a->getValue(g_).type_;
	if (t!=Value::INT && t!=Value::STRING && t!=Value::FUNCSYMBOL)
		t=values[idx].type_;

	switch(t)
	{	case Value::INT:
			return(true);
		case Value::FUNCSYMBOL:
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug)
{	std::cerr << "arg is FuncSymbol: ";
	g_->getFuncSymbol(a->getValue(g_).uid_)->print(g_,std::cerr);
	std::cerr << std::endl;
}
#endif
			uid=g_->getFuncSymbol(a->getValue(g_).uid_)->getName();
			arity=g_->getFuncSymbol(a->getValue(g_).uid_)->getValues().size();
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg's functor uid=" << uid << "; arity=" << arity << std::endl;
#endif
			break;
		case Value::STRING:
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg is String" << std::endl;
#endif
			uid=a->getValue(g_).uid_;
			arity=0;
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "arg's string functor uid=" << uid << "; arity=" << arity << std::endl;
#endif
			break;
		default:
			std::cerr << "internal error: unknown term type " << t << std::endl;
			exit(1);
	}

	return(g_->getCSPSolver()->isHidden(uid,arity));
}

int TAtomPredicateLiteral::nTerms(const ValueVector &values)
{	int term_counter;

	term_counter=0;
	for (int i=0;i<(*variables_).size();i++)
		if (isFTerm((*variables_)[i],i,values)) term_counter++;

#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "nTerms=" << term_counter << std::endl;
#endif

	return(term_counter);
}

NS_OUTPUT::Object * TAtomPredicateLiteral::convert(const ValueVector &values)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tatompred's convert()" << std::endl;
#endif
	Term *a_, *b_;
	Clasp::CSPConstraint constraint_;

	a_=(*variables_)[0];
	b_=(*variables_)[1];

	constraint_ = Clasp::CSPConstraint();

	Clasp::CSPConstraint* ac;
	if (isUndefined(a_))
		ac=getCSPConstraintOfValue(g_,values[0]);	/* the value from the grounder is needed */
	else
		ac = a_->getCSPConstraint(g_);
	ac->setFTerm(isFTerm(a_,0,values));
	if (isShown(a_,0,values))
		ac->setVisible(true);
	else
	if (isHidden(a_,0,values))
		ac->setVisible(false);

	Clasp::CSPConstraint* bc;
	if (isUndefined(b_))
		bc=getCSPConstraintOfValue(g_,values[1]);	/* the value from the grounder is needed */
	else
		bc = b_->getCSPConstraint(g_);
	bc->setFTerm(isFTerm(b_,1,values));
	if (isShown(b_,1,values))
		bc->setVisible(true);
	else
	if (isHidden(b_,1,values))
		bc->setVisible(false);

	if (ac==NULL || bc==NULL)
		throw GrinGoException("error: internal error. Probably an unsafe expression containing t-literals.");

//	std::string s(a_->getCTerm(g_).getString());
//	std::string s(values[0].asString(g_));
//std::cerr << "str s is " << s << std::endl;
	switch(type_)
	{
		case EQ:
			constraint_.setRelation(Clasp::CSPConstraint::EQ, ac, bc);
			break;
		case NE:
			constraint_.setRelation(Clasp::CSPConstraint::NE, ac, bc);
			break;
		case GT:
			constraint_.setRelation(Clasp::CSPConstraint::GT, ac, bc);
			break;
		case GE:
			constraint_.setRelation(Clasp::CSPConstraint::GE, ac, bc);
			break;
		case LE:
			constraint_.setRelation(Clasp::CSPConstraint::LE, ac, bc);
			break;
		case LT:
			constraint_.setRelation(Clasp::CSPConstraint::LT, ac, bc);
			break;
		default:
			assert(false);
	}

	int n;
	if (headTAtom_ && (n=nTerms(values))!=1)
	{	if (n>1)
			throw GrinGoException("error: dependent t-atoms cannot occur in the head of rules");
		else
			throw GrinGoException("error: t-atoms in the head of rules must contain at least one non-Herbrand term");
	}

#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) 
{	std::cerr << "convert() using values:" << std::endl;
	for(int i=0;i<values.size();i++)
	{	std::cerr << "  val(" << i << ")=" << values[i].asString(g_) << std::endl;
	}
}
#endif
	return new NS_OUTPUT::CSPAtom(getNeg(), predNode_, uid_, values, constraint_);
}


NS_OUTPUT::Object *TAtomPredicateLiteral::convert()
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "tatompred's convert2" << std::endl;
#endif
	// the method isFact has to be called before this method
	return convert(values_);
}

/*
void TAtomPredicateLiteral::addDomain(ValueVector &values)
{
std::cerr << "pred's addDomain" << std::endl;
	predNode_->addDomain(values);
}
*/

/*
IndexedDomain *PredicateLiteral::createIndexedDomain(Grounder *g, VarSet &index)
{
std::cerr << "pred's createIndexedDomain" << std::endl;
	if(!predNode_->complete() || getNeg())
		return new IndexedDomainMatchOnly(this);
	if(variables_)
	{
		VarSet vars, free;
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
			(*it)->getVars(vars);
		for(VarSet::iterator it = vars.begin(); it != vars.end(); it++)
			if(index.find(*it) == index.end())
				free.insert(*it);
		if(free.size() > 0)
		{
			return new IndexedDomainNewDefault(g, predNode_->getDomain(), index, *variables_);
		}
		else
			return new IndexedDomainMatchOnly(this);
	}
	else
		return new IndexedDomainMatchOnly(this);
}
*/
Literal* TAtomPredicateLiteral::clone() const
{
	return new TAtomPredicateLiteral(*this);
}

IndexedDomain *TAtomPredicateLiteral::createIndexedDomain(Grounder *g, VarSet &index)
{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "TAtomPredicateLiteral::createIndexedDomain() " << this << std::endl;
#endif

	if (type_!=EQ)	/* connectives other than #= must be treated just like "not l" */
		return new IndexedDomainMatchOnly(this);

	return(PredicateLiteral::createIndexedDomain(g,index));
}

double TAtomPredicateLiteral::heuristicValue()
{
	if (type_!=EQ)
	{
#ifdef ASPF_DEBUG_LIGHT
if (g_->options().debug) std::cerr << "Not #=. Heuristic value for grounding is the highest, because it cannot be used for grounding just like 'not l'" << std::endl;
#endif
		return DBL_MAX;
	}
	else
		return(PredicateLiteral::heuristicValue());
}

TAtomPredicateLiteral::~TAtomPredicateLiteral()
{
//	if(a_)
//		a_=NULL;	/* deallocated as part of variables_ by PredicateLiteral */
//	if(b_)
//		b_=NULL;	/* deallocated as part of variables_ by PredicateLiteral */
}

/*
Domain *TAtomPredicateLiteral::getDomain() const
{
	return predNode_;
}
*/

/*
void TAtomPredicateLiteral::preprocess(Grounder *g, Expandable *e)
{
std::cerr << "pred's preprocess" << std::endl;
	if(variables_)
	{
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
			(*it)->preprocess(this, *it, g, e);
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
			if((*it)->isComplex())
			{
				int var = g->createUniqueVar();
				e->appendLiteral(new AssignmentLiteral(new Variable(g, var), *it), Expandable::COMPLEXTERM);
				*it = new Variable(g, var);
			}
	}
	if((*g->getString(id_))[0] == '-')
		g->addTrueNegation(id_, variables_ ? variables_->size() : 0);
}
*/

/*
TermVector *TAtomPredicateLiteral::getArgs()
{
	return variables_;
}


int TAtomPredicateLiteral::getUid()
{
	return uid_;
}

int TAtomPredicateLiteral::getId()
{
	return id_;
}

int TAtomPredicateLiteral::getArity() const
{
	return variables_->size();
}
*/
