// Copyright (c) 2008, Roland Kaminski
//
// This file is part of GrinGo.
//
// GrinGo is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// GrinGo is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with GrinGo.  If not, see <http://www.gnu.org/licenses/>.

#include "statementdependencygraph.h"
#include "literaldependencygraph.h"
#include "statementchecker.h"
#include "predicateliteral.h"
#include "assignmentliteral.h"
#include "relationliteral.h"
#include "grounder.h"
#include "variable.h"
#include "term.h"
#include "domain.h"
#include "rangeterm.h"
#include "rangeliteral.h"
#include "indexeddomain.h"
#include "dlvgrounder.h"
#include "value.h"
#include "multipleargsterm.h"
#include "output.h"
#include "evaluator.h"
#include "bindersplitter.h"

using namespace NS_GRINGO;
		
PredicateLiteral::PredicateLiteral(Grounder *g, int id, TermVector *variables) : Literal(), uid_(g->createPred(id, variables->size())), predNode_(g->getDomain(uid_)), id_(id), variables_(variables), values_(variables ? variables->size() : 0)
{
}

PredicateLiteral::PredicateLiteral(const PredicateLiteral &p) : Literal(p), uid_(p.uid_), predNode_(p.predNode_), id_(p.id_), values_(p.values_.size())
{
        if(p.variables_)
        {
                variables_ = new TermVector();
                for(TermVector::iterator it = p.variables_->begin(); it != p.variables_->end(); it++)
                                variables_->push_back((*it)->clone());
        }
        else
                variables_ = 0;
}

bool PredicateLiteral::checkO(LiteralVector &unsolved)
{
	return true;
}

SDGNode *PredicateLiteral::createNode(SDG *dg, SDGNode *prev, DependencyAdd todo)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::createNode1() " << this << std::endl;
#endif
	SDGNode *n = dg->createPredicateNode(this);
	switch(todo)
	{
		case ADD_BODY_DEP:
			prev->addDependency(n, getNeg());
			break;
		case ADD_HEAD_DEP:
			n->addDependency(prev);
			break;
		case ADD_NOTHING:
			break;
	}
	return n;
}

void PredicateLiteral::createNode(LDGBuilder *dg, bool head)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::createNode2() " << this << std::endl;
#endif
	VarSet needed, provided;
	if(head || getNeg() || !predNode_->complete())
	{
		if(variables_)
			for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
				(*it)->getVars(needed);
	}
	else
	{
		if(variables_)
			for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
				if((*it)->isComplex())
					(*it)->getVars(needed);
				else
					(*it)->getVars(provided);
	}
	dg->createNode(this, head, needed, provided);
}

void PredicateLiteral::createNode(StatementChecker *dg, bool head, bool delayed)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::createNode3() " << this << std::endl;
#endif
	VarSet vars, empty;
	getVars(vars);
	if(head || getNeg() || !predNode_->complete())
		dg->createNode(vars, empty);
	else
		dg->createNode(empty, vars);
}

void PredicateLiteral::print(const GlobalStorage *g, std::ostream &out) const
{
	if(getNeg())
		out << "not ";
	out << *g->getString(id_);
	if(getArity() > 0)
	{
		out << "(";
		bool comma = false;
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
		{
			if(comma)
				out << ", ";
			else
				comma = true;
			out << pp(g, *it);
		}
		out << ")";
	}
}

const ValueVector &PredicateLiteral::getValues()
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::getValues() "<< this << std::endl;
#endif
	return values_;
}

void PredicateLiteral::reset()
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::reset() " << this << std::endl;
#endif
	if(!getNeg())
		predNode_->reset();
}

void PredicateLiteral::finish()
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::finish() " << this << std::endl;
#endif
	if(!getNeg())
		predNode_->finish();
}

void PredicateLiteral::evaluate()
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::evaluate() " << this << std::endl;
#endif
	if(!getNeg())
		predNode_->evaluate();
}

bool PredicateLiteral::isFact(const ValueVector &values)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::isFact(const ValueVector &values)1" << std::endl;
#endif
	// a precondition for this method is that the predicate is not false!
	if(predNode_->solved())
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact1a" << std::endl;
#endif
		return true;
#ifdef ASPF_DEBUG
}
#endif
	if(getNeg())
	{
		assert(!predNode_->isFact(values));
		if(predNode_->complete())
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact1b " << !predNode_->inDomain(values) << std::endl;
#endif
			return !predNode_->inDomain(values);
#ifdef ASPF_DEBUG
}
#endif
		else
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact1c" << std::endl;
#endif
			return false;
#ifdef ASPF_DEBUG
}
#endif
	}
	else
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact1d " << predNode_->isFact(values) << std::endl;
#endif
		return predNode_->isFact(values);
#ifdef ASPF_DEBUG
}
#endif
}

bool PredicateLiteral::isFact(Grounder *g)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::isFact(const ValueVector &values)2" << std::endl;
#endif
	// a precondition for this method is that the predicate is not false!
	if(predNode_->solved())
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact2a" << std::endl;
#endif
		return true;
#ifdef ASPF_DEBUG
}
#endif
	for(int i = 0; i < (int)variables_->size(); i++)
		values_[i] = (*variables_)[i]->getValue(g);
	if(getNeg())
	{
		assert(!predNode_->isFact(values_));
		if(predNode_->complete())
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact2b " << !predNode_->inDomain(values_) << std::endl;
#endif
			return !predNode_->inDomain(values_);
#ifdef ASPF_DEBUG
}
#endif
		else
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact2c" << std::endl;
#endif
			return false;
#ifdef ASPF_DEBUG
}
#endif
	}
	else
#ifdef ASPF_DEBUG
{
std::cerr << "  isFact2d " << predNode_->isFact(values_) << std::endl;
#endif
		return predNode_->isFact(values_);
#ifdef ASPF_DEBUG
}
#endif
}

bool PredicateLiteral::solved()
{
	return predNode_->solved();
}

bool PredicateLiteral::match(const ValueVector &values)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::match()1" << std::endl;
#endif
	if(!predNode_->complete())
	{
		if(getNeg())
#ifdef ASPF_DEBUG
{
std::cerr << "match()1a " << !predNode_->isFact(values) << std::endl;
#endif
			return !predNode_->isFact(values);
#ifdef ASPF_DEBUG
}
#endif
		else
#ifdef ASPF_DEBUG
{
std::cerr << "match()1b" << std::endl;
#endif
			return true;
#ifdef ASPF_DEBUG
}
#endif
	}
	bool match = predNode_->inDomain(values);
	if(getNeg())
	{
#ifdef ASPF_DEBUG
{
std::cerr << "match()1c " << !(match && (predNode_->solved() || predNode_->isFact(values))) << std::endl;
#endif
		return !(match && (predNode_->solved() || predNode_->isFact(values)));
#ifdef ASPF_DEBUG
}
#endif
	}
	else
	{
#ifdef ASPF_DEBUG
std::cerr << "match()1d " << match << std::endl;
#endif
		return match;
	}
}

bool PredicateLiteral::match(Grounder *g)
{
	assert(!predNode_->solved() || predNode_->complete());
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::match()2" << std::endl;
#endif
	// incomplete prediactes match everything
	// except if they contain some values that are facts!!!
	if(!predNode_->complete())
	{
		if(getNeg() && predNode_->hasFacts())
		{
			for(int i = 0; i < (int)variables_->size(); i++)
				values_[i] = (*variables_)[i]->getValue(g);
#ifdef ASPF_DEBUG
std::cerr << "match()2a " << !predNode_->isFact(values_) << std::endl;
#endif
			return !predNode_->isFact(values_);
		}
		else
#ifdef ASPF_DEBUG
{
std::cerr << "match()2b" << std::endl;
#endif
			return true;
#ifdef ASPF_DEBUG
}
#endif
	}
	// check if current assignment is satisfied wrt the domain
	bool match = true;
	if(variables_)
	{
		for(int i = 0; i < (int)variables_->size(); i++)
			values_[i] = (*variables_)[i]->getValue(g);
	}
	match = predNode_->inDomain(values_);
	if(getNeg())
	{
#ifdef ASPF_DEBUG
std::cerr << "match()2c " << !(match && (predNode_->solved() || predNode_->isFact(values_))) << std::endl;
#endif
		return !(match && (predNode_->solved() || predNode_->isFact(values_)));
	}
	else
	{
#ifdef ASPF_DEBUG
std::cerr << "match()2d " << match << std::endl;
#endif
		return match;
	}
}

NS_OUTPUT::Object * PredicateLiteral::convert(const ValueVector &values)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::convert()1" << std::endl;
#endif
	return new NS_OUTPUT::Atom(getNeg(), predNode_, uid_, values);
}

NS_OUTPUT::Object *PredicateLiteral::convert()
{
	// the method isFact has to be called before this method
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::convert()2" << std::endl;
#endif
	return convert(values_);
}

void PredicateLiteral::addDomain(ValueVector &values)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::addDomain()" << std::endl;
#endif
	predNode_->addDomain(values);
}

IndexedDomain *PredicateLiteral::createIndexedDomain(Grounder *g, VarSet &index)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::createIndexedDomain() " << this << " rel=" << *g->getString(id_) << std::endl;
#endif
	if(!predNode_->complete() || getNeg())
		return new IndexedDomainMatchOnly(this);
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::createIndexedDomain()2 " << this << std::endl;
#endif
	if(variables_)
	{
		VarSet vars, free;
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
			(*it)->getVars(vars);
		for(VarSet::iterator it = vars.begin(); it != vars.end(); it++)
			if(index.find(*it) == index.end())
				free.insert(*it);
		if(free.size() > 0)
		{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::createIndexedDomain()3 " << this << std::endl;
#endif
			return new IndexedDomainNewDefault(g, predNode_->getDomain(), index, *variables_);
		}
		else
#ifdef ASPF_DEBUG
{
std::cerr << "PredicateLiteral::createIndexedDomain()4 " << this << std::endl;
#endif
			return new IndexedDomainMatchOnly(this);
#ifdef ASPF_DEBUG
}
#endif
	}
	else
#ifdef ASPF_DEBUG
{
std::cerr << "PredicateLiteral::createIndexedDomain()5 " << this << std::endl;
#endif
		return new IndexedDomainMatchOnly(this);
#ifdef ASPF_DEBUG
}
#endif
}

Literal* PredicateLiteral::clone() const
{
	return new PredicateLiteral(*this);
}

PredicateLiteral::~PredicateLiteral()
{
	if(variables_)
	{
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
		{
			delete *it;
		}
		delete variables_;
	}
}

Domain *PredicateLiteral::getDomain() const
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::getDomain() " << this << std::endl;
#endif
	return predNode_;
}

void PredicateLiteral::preprocess(Grounder *g, Expandable *e)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::preprocess() " << this << std::endl;
#endif
	if(variables_)
	{
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
			(*it)->preprocess(this, *it, g, e);
		for(TermVector::iterator it = variables_->begin(); it != variables_->end(); it++)
			if((*it)->isComplex())
			{
				int var = g->createUniqueVar();
				e->appendLiteral(new AssignmentLiteral(new Variable(g, var), *it), Expandable::COMPLEXTERM);
				*it = new Variable(g, var);
			}
	}
	if((*g->getString(id_))[0] == '-')
		g->addTrueNegation(id_, variables_ ? variables_->size() : 0);
}

TermVector *PredicateLiteral::getArgs()
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::getArgs() " << this << std::endl;
#endif
	return variables_;
}


int PredicateLiteral::getUid()
{
	return uid_;
}

int PredicateLiteral::getId()
{
	return id_;
}

int PredicateLiteral::getArity() const
{
	return variables_->size();
}

double PredicateLiteral::heuristicValue()
{
	// TODO: thats too simple !!! i want something better
	if(!getNeg())
	{
		if(predNode_->complete())
		{
			if(variables_ && variables_->size() > 0)
				return pow(predNode_->getDomain().size(), 1.0 / variables_->size());
			else
				return 0;
		}
		else
			// the literal doenst help at all
			return DBL_MAX;
	}
	else if(predNode_->solved())
	{
		// the literal can be used to check if the assignment is valid
		return 0;
	}
	else
		return DBL_MAX;
}

void PredicateLiteral::addIncParam(Grounder *g, const Value &v)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::addIncParam() " << this << std::endl;
#endif
	if(variables_)
		for(TermVector::iterator i = variables_->begin(); i != variables_->end(); i++)
			(*i)->addIncParam(g, *i, v);
}

void PredicateLiteral::getVars(VarSet &vars) const
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::getVars() " << this << std::endl;
#endif
	if(variables_)
		for(TermVector::const_iterator it = variables_->begin(); it != variables_->end(); it++)
			(*it)->getVars(vars);
}

void PredicateLiteral::binderSplit(Expandable *e, const VarSet &relevant)
{
#ifdef ASPF_DEBUG
std::cerr << "PredicateLiteral::binderSplit() " << this << std::endl;
#endif
	if(!variables_ || getNeg() || !solved())
		return;
	VarVector r;
	VarSet vars;
	getVars(vars);
	for(VarSet::iterator it = vars.begin(); it != vars.end(); it++)
		if(relevant.find(*it) != relevant.end())
			r.push_back(*it);
	if(r.size() != vars.size() && r.size() > 0)
	{
		//std::cerr << "bindersplit: " << this << std::endl;
		e->appendLiteral(new BinderSplitter(predNode_, variables_, r), Expandable::RANGETERM);
	}
}

