// 
// Copyright (c) 2006-2007, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#include <clasp/include/cspconstraint.h>
#include <clasp/include/util/misc_types.h>
#include <cassert>

namespace Clasp {

CSPConstraint::CSPConstraint() : type_(UNDEF), var_(0), a_(0), b_(0), lin_(0)
{
	isFTerm_=false;
	isShown_=false;
	isHidden_=false;
}

CSPConstraint::~CSPConstraint()
{
	clear();
}


const CSPConstraint CSPConstraint::operator=(const CSPConstraint& cc)
{
	clear();
	type_ = cc.type_;
	var_ = cc.var_;
	a_ = cc.a_ ? new CSPConstraint(*cc.a_) : 0;
	b_ = cc.b_ ? new CSPConstraint(*cc.b_) : 0;
	lin_ = cc.lin_;
	isFTerm_=cc.isFTerm_;
	isShown_=cc.isShown_;
	isHidden_=cc.isHidden_;
	return *this;
}


CSPConstraint::CSPConstraint(const CSPConstraint& cc) : type_(cc.type_), var_(cc.var_),
                                                        a_(cc.a_ ? new CSPConstraint(*cc.a_) : 0),
                                                        b_(cc.b_ ? new CSPConstraint(*cc.b_) : 0),
																		  lin_(cc.lin_)
{
	isFTerm_=cc.isFTerm_;
	isShown_=cc.isShown_;
	isHidden_=cc.isHidden_;

}

void CSPConstraint::setVariable(unsigned int var)
{
	type_ = VARIABLE;
	var_ = var;
	lin_ = 1;
}

void CSPConstraint::setInteger(int i)
{
	type_ = INTEGER;
	integer_ = i;
	lin_ = 1;
}


void CSPConstraint::setOperator(Operator op, CSPConstraint* a, CSPConstraint* b)
{
	type_ = OPERATOR;
	op_ = op;
	a_ = a;
	b_ = b;
	if (op == PLUS || op == MINUS)
		lin_ = a->lin_ + b->lin_;
	else
	if (op == ABS)
		lin_ = 1;
	else
	if (op == TIMES || op == DIVIDE)
	{
		if (a->getType() == INTEGER)
			lin_ = b->lin_;
		else
		if (b->getType() == INTEGER)
			lin_ = a->lin_;
		else
			lin_ = 1;
	}
}

unsigned int CSPConstraint::getLinearSize() const
{
	return lin_;
}

void CSPConstraint::setRelation(Relation rel, CSPConstraint* a, CSPConstraint* b)
{
	type_ = RELATION;
	rel_ = rel;
	a_ = a;
	b_ = b;
	lin_ = a->lin_ + b->lin_;
}


void CSPConstraint::clear()
{
	delete a_;
	delete b_;
	a_ = 0;
	b_ = 0;
}


CSPConstraint::Type CSPConstraint::getType() const
{
	return type_;
}
			

unsigned int CSPConstraint::getVar() const
{
	assert(type_ == VARIABLE);
	return var_;
}


int CSPConstraint::getInteger() const
{
	assert(type_ == INTEGER);
	return integer_;
}


CSPConstraint::Operator CSPConstraint::getOperator(CSPConstraint*& a, CSPConstraint*& b) const
{
	assert(type_ == OPERATOR);

	a = a_;
	b = b_;
	return op_;
}


CSPConstraint::Relation CSPConstraint::getRelation(CSPConstraint*& a, CSPConstraint*& b) const
{
	assert(type_ == RELATION);

	a = a_;
	b = b_;
	return rel_;
}


std::vector<unsigned int> CSPConstraint::getAllVariables()
{
	std::vector<unsigned int> re;
	if (type_ == VARIABLE)
		re.push_back(var_);
	if (type_ == OPERATOR || type_ == RELATION)
	{
		if (a_)
		{
			std::vector<unsigned int> v(a_->getAllVariables());
			re.insert(re.end(), v.begin(), v.end());
		}
		if (b_)
		{
			std::vector<unsigned int> v(b_->getAllVariables());
			re.insert(re.end(), v.begin(), v.end());
		}
	}
	return re;

}

}

