// 
// Copyright (c) 2006-2007, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#include <iostream>
#include <clasp/include/aspfsolver.h>
#include <clasp/include/program_builder.h>
#include <clasp/include/clause.h>
#include <exception>

#define ASPF_DEBUG_LIGHT
bool debug_;

namespace Clasp {

	class ASPmCSPException : public std::exception
	{
	public:
		ASPmCSPException(const std::string &message) : message_(message)
		{
		}
		const char* what() const throw()
		{
			return message_.c_str();
		}
		virtual ~ASPmCSPException() throw()
		{
		}
	private:
		const std::string message_;
	};

//#define DEBUGTEXT

#define USE_LEFT_TO_RIGHT_NOGOODS

ASPfSolver::ASPfSolver(bool lazyLearn, bool useCDG, bool weakAS, int numAS, bool debug)
{
/* ASPfSolver stuff */
assignments=NULL;
is_assigned=NULL;
assignment_level=NULL;
n_terms=0;
hideAll_=false;
ftermAll_=false;
debug_=debug;
/* */
}

ASPfSolver::~ASPfSolver()
{
}

std::string ASPfSolver::toString(TAtomValue v)
{	std::stringstream s;

	s << "<" << (v.first ? "true":"false") << "," << v.second << ">";

	return(s.str());
}

/* ASPf stuff [marcy 011712] */

void ASPfSolver::displayTermExpression(TermExpression *c)
{	TermExpression *c1,*c2;

	const char *rels[]=
	{	"=",
		"!=",
		"<=",
		"<",
		">=",
		">"
	};
	const char *ops[]=
	{	"abs",
		"+",
		"-",
		"*",
		"/"
	};

	switch(c->getType())
	{	case CSPConstraint::RELATION:
			std::cerr << "relation: " << rels[c->getRelation(c1,c2)] << std::endl;
			std::cerr << " a: "; displayTermExpression(c1); std::cerr << std::endl;
			std::cerr << " b: "; displayTermExpression(c2); std::cerr << std::endl;
			break;
		case CSPConstraint::OPERATOR:
			std::cerr << "operator: " << ops[c->getOperator(c1,c2)] << std::endl;
			std::cerr << " a: "; displayTermExpression(c1); std::cerr << std::endl;
			std::cerr << " b: "; displayTermExpression(c2); std::cerr << std::endl;
			break;
		case CSPConstraint::VARIABLE:
			//std::cerr << "var(" << c->getVar() << "; "<< variables_[c->getVar()] << ")";
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "var # is " << c->getVar() << std::endl;
#endif
			std::cerr << variables_[c->getVar()];
			break;
		case CSPConstraint::INTEGER:
			std::cerr << "int(" << c->getInteger() << ")";
			break;
		default:
			std::cerr << "unknown CSPConstraint type " << c->getType() << std::endl;
			break;
	}
}

void ASPfSolver::growAssignments(TermExpression *c)
{	TermExpression *c1,*c2;
	int term;

	switch(c->getType())
	{	case CSPConstraint::RELATION:
			c->getRelation(c1,c2);
			growAssignments(c1);
			growAssignments(c2);
			break;
		case CSPConstraint::OPERATOR:
			c->getOperator(c1,c2);
			growAssignments(c1);
			growAssignments(c2);
			break;
		case CSPConstraint::VARIABLE:
			term=c->getVar();
			if (term>=n_terms)	/* first term has term=0 */
			{	if (n_terms>0)
				{	free(assignments);
					free(is_assigned);
					free(assignment_level);
				}
				assignments=(TAtomValue *)calloc(term+1,sizeof(TAtomValue));
				is_assigned=(bool *)calloc(term+1,sizeof(bool));
				assignment_level=(int *)calloc(term+1,sizeof(int));
				n_terms=term+1;
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "n_terms now " << n_terms << std::endl;
#endif
			}
			break;
		case CSPConstraint::INTEGER:
			break;
		default:
			break;
	}
}

ASPfSolver::ASPfConstraint::ASPfConstraint(ASPfSolver* aspf,TAtom *tatom,LitVec& reason)
	: aspf_(aspf), tatomName_(tatom->name())
{	for (LitVec::const_iterator j = reason.begin(); j != reason.end(); ++j)
		reason_.push_back(*j);
}

ASPfSolver::ASPfConstraint *ASPfSolver::ASPfConstraint::createStandard(ASPfSolver *aspf_,TAtom *tatom,bool useSelf,bool reasonTruthValue)
{	std::vector<unsigned int> terms;
	LitVec reason;

#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	std::cerr << "in ASPfSolver::ASPfConstraint::createStandard" << std::endl;
	std::cerr << "Generating Reason with the seed t-atoms of " << tatom->name() <<  ", useSelf=" << useSelf << ", reasonTruthValue=" << reasonTruthValue << ":" << std::endl;
	std::cerr << "isTrue=" << (int)aspf_->s_->isTrue(tatom->literal()) << "; isFalse=" << (int)aspf_->s_->isFalse(tatom->literal()) << std::endl;
}
#endif

	if (useSelf)
		reason.push_back(tatom->literal());

	terms=tatom->getAllTerms();
	for (std::vector<unsigned int>::iterator j = terms.begin(); j != terms.end(); ++j)
	{	int term=*j;

#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	std::cerr << " term #=" << term << std::endl;
	std::cerr << " reasons: ";
}
#endif
		for (std::vector<TAtom *>::const_iterator sPtr = aspf_->seedTAtomsForTerm[term].begin(); sPtr != aspf_->seedTAtomsForTerm[term].end(); ++sPtr)
		{	TAtom *s;
			bool addme;
		
			s=*sPtr;

			if (aspf_->is_assigned[term])	/* if term is assigned a value, then let's compare values */
			{	addme=((reasonTruthValue==true && aspf_->assignments[term]==s->seedTAtomValue()) ||	/* use as reason if the seed t-atom has the same value as currently assigned to the term */
				       (reasonTruthValue==false && aspf_->assignments[term]!=s->seedTAtomValue()));    /* use as reason if the seed t-atom has value different what currently assigned to the term */
			}
			else			/* otherwise (but I don't think we should ever get here) consider truth of the literals */
			addme=(s->literal()!=tatom->literal() &&
			       ((reasonTruthValue==true && aspf_->s_->isTrue(s->literal())) ||
			        (reasonTruthValue==false && aspf_->s_->isFalse(s->literal()))));
			if (addme)
			{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << (aspf_->s_->isTrue(s->literal()) ? "":"not ") << s->name() << " ";
#endif
				reason.push_back(s->literal());
			}
		}
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << std::endl;
#endif
	}

	if (reason.size()==0)
		return(0);	/* no reason given */

	ASPfConstraint *r=new ASPfConstraint(aspf_,tatom,reason);
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfConstraint ptr=" << r << std::endl;
#endif
	return(r);
}

void ASPfSolver::ASPfConstraint::reason(const Literal& l, LitVec& reason)
{	TAtom *tatom;
	CSPConstraint *c;
	std::vector<unsigned int> terms;

#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	std::cerr << "in ASPfSolver::ASPfConstraint::reason()!!!!" << std::endl;
	std::cerr << "  Offering Reason for " << tatomName_ << ":" << std::endl;
	std::cerr << "  isTrue=" << (int)aspf_->s_->isTrue(l) << "; isFalse=" << (int)aspf_->s_->isFalse(l) << std::endl;
	std::cerr << "  this=" << this << std::endl;
}
#endif

	for (LitVec::const_iterator j = reason_.begin(); j != reason_.end(); ++j)
	{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "  var=" << j->var() << " trueValue=" << (int)trueValue(*j) << std::endl;
#endif

		reason.push_back(*j);
	}
}

/* END of ASPf stuff [marcy 011712] */

void ASPfSolver::addConstraint(CSPConstraint c, int uid)
{
/* ASPfSolver stuff */
#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	std::cerr << "addConstraint:" <<std::endl;
	std::cerr << "uid=" << uid << std::endl;;
	std::cerr << "t-atom=";
	displayTermExpression(&c);
	std::cerr << "------------------" << std::endl;
}
#endif
	growAssignments(&c);
/* */
	tatoms_[uid].push_back(new TAtom(c));
}

/*			
void ASPfSolver::setDomain(int lower, int upper)
{
	if (addedDomain_)
		throw ASPmCSPException("Only one domain predicate allowed!");

	domain_.first = lower;
	domain_.second = upper;
	addedDomain_ = true;
}
*/

void ASPfSolver::setSolver(Solver* s)
{
	assert(s);
	s_ = s;
	s_->setCSPSolver(this);
}

void ASPfSolver::setFTerm(int id, int arity)
{	fterms_.push_back(std::make_pair(id,arity));

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::setFTerm(" << id << "," << arity << ")" << std::endl;
#endif
}

bool ASPfSolver::isFTerm(int id, int arity)
{	if (ftermAll_) return(true);

	for (int i=0;i<fterms_.size();i++)
	{	if (fterms_[i]==std::make_pair(id,arity))
			return(true);
	}

	return(false);
}

void ASPfSolver::setVisible(int id, int arity, bool visible)
{
//std::cerr << "WARNING: ignoring arg visible in ASPfSolver::setVisible" << std::endl;
//	visibles_.push_back(std::make_pair(id,arity));
	if (visible)
		shown_.push_back(std::make_pair(id,arity));
	else
		hidden_.push_back(std::make_pair(id,arity));
}

bool ASPfSolver::isShown(int id, int arity)
{	for (int i=0;i<shown_.size();i++)
	{	if (shown_[i]==std::make_pair(id,arity))
			return(true);
	}

	return(false);
}

bool ASPfSolver::isHidden(int id, int arity)
{	for (int i=0;i<hidden_.size();i++)
	{	if (hidden_[i]==std::make_pair(id,arity))
			return(true);
	}

	return(false);
}

/* [marcy 011712] */
bool ASPfSolver::getValueOfTermExpression(TermExpression *c,TAtomValue &val)
{	TermExpression *c1,*c2;
	int op;
	TAtomValue v1,v2;
	bool ret;

	ret=false;
	switch(c->getType())
	{	case CSPConstraint::RELATION:
			std::cerr << "internal error: relation in the scope of a non-Herbrand relation!" << std::endl;
			exit(1);
		case CSPConstraint::OPERATOR:
			op=c->getOperator(c1,c2);
			if (getValueOfTermExpression(c1,v1) && getValueOfTermExpression(c2,v2))
			{
				/* [marcy 041112] */
				if (v1.first==true || v2.first==true)
				{	std::cerr << "error: arithmetic operator " << op << " applied to non-arithmetic constants "; displayTermExpression(c1); std::cerr << " and "; displayTermExpression(c2); std::cerr << ". Aborting." << std::endl;
					exit(1);
				}
				/* */
				
				int i1,i2,iv;
				i1=v1.second;
				i2=v2.second;
				switch(op)
				{	case CSPConstraint::ABS:
						std::cerr << "internal error: abs() not correctly handled yet " << op << std::endl;	// FIXME!!!!
						exit(1);
					case CSPConstraint::PLUS:
						iv=i1+i2;
						break;
					case CSPConstraint::MINUS:
						iv=i1-i2;
						break;
					case CSPConstraint::TIMES:
						iv=i1*i2;
						break;
					case CSPConstraint::DIVIDE:
						iv=i1/i2;
						break;
					default:
						std::cerr << "internal error: unknown operator " << op << std::endl;
						exit(1);
				}
				val=std::make_pair(false,iv);
				ret=true;
			}
			break;
		case CSPConstraint::VARIABLE:
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "getValueOfConstraint(): variable index " << c->getVar() << ": is_assigned=" << is_assigned[c->getVar()] << " val=" << toString(assignments[c->getVar()]) << std::endl;
#endif
/* [marcy 041112] */
			if (!c->isFTerm())
			{	
				val=std::make_pair(true,c->getVar());
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "getValueOfConstraint(): treating expression as non-arithmetic constant instead of f-term; val=" << toString(val) << std::endl;
#endif
				ret=true;
			}
			else
/* */
			if (is_assigned[c->getVar()])
			{	val=assignments[c->getVar()];
				ret=true;
			}
			break;
		case CSPConstraint::INTEGER:
			/* [marcy 041112] */
			//val=c->getInteger();
			val=std::make_pair(false,c->getInteger());
			/* */
			ret=true;
			break;
		default:
			std::cerr << "internal error: unknown TermExpression type " << c->getType() << std::endl;
			exit(1);
	}
	return(ret);
}

/*
bool ASPfSolver::isSeedTAtom(TAtom *c)
{	TermExpression *c1,*c2;

	return (c->getType()==CSPConstraint::RELATION &&
		c->getRelation(c1,c2)==CSPConstraint::EQ && 
		c1->getType()==CSPConstraint::VARIABLE &&
		c2->getType()==CSPConstraint::INTEGER);
}
*/

bool ASPfSolver::isSatisfied(TAtomValue v1,int rel,TAtomValue v2)
{
	bool satisfied;

	if (rel!=CSPConstraint::EQ && rel!=CSPConstraint::NE)
	{
		if (v1.first!=v2.first)
		{	std::cerr << "ERROR: trying to compare arithmetic constant with non-arithmetic constant MAKES NO SENSE RIGHT NOW!!! Aborting." << std::endl;
			exit(1);
		}
		if (v1.first)
			std::cerr << "WARNING: comparison between non-arithmetic constants should be lexicographic but is NOT!!!" << std::endl;
	}

	switch(rel)
	{	case CSPConstraint::EQ:
			satisfied=v1==v2;
			break;
		case CSPConstraint::NE:
			satisfied=v1!=v2;
			break;
		case CSPConstraint::LE:
			satisfied=v1<=v2;
			break;
		case CSPConstraint::LT:
			satisfied=v1<v2;
			break;
		case CSPConstraint::GE:
			satisfied=v1>=v2;
			break;
		case CSPConstraint::GT:
			satisfied=v1>v2;
			break;
		default:
			std::cout << "INTERNAL ERROR: unknown comparison operator " << rel << std::endl;
			exit(1);
	}

	return(satisfied);
}

/*
 * \pre: all the terms of *c have been assigned a value
 *
 * Returns true if the t-atom is satisfied by the current
 * value assignment, and false otherwise.
 */
bool ASPfSolver::isTAtomSatisfied(TAtom *c)
{	int rel;
	TAtomValue v1,v2;
	bool satisfied;
	TermExpression *c1,*c2;

	if (!c->getType()==CSPConstraint::RELATION)
	{	std::cout << "INTERNAL ERROR: a t-atom does not have a relation in it???" << std::endl;
		exit(1);
	}

	rel=c->getRelation(c1,c2);

	getValueOfTermExpression(c1,v1);
	getValueOfTermExpression(c2,v2);

	satisfied=isSatisfied(v1,rel,v2);

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "isTAtomSatisfied returning " << (int)satisfied << std::endl;
#endif

	return(satisfied);
}

void ASPfSolver::exploreRecursivelyClosure(TAtom *tatom,int term_idx,std::vector<unsigned int> &terms,TAtomValue *common_value,bool *in_common,int &n_in_common,bool *value_assigned)
{	int term;
	int i;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "in ASPfSolver::exploreRecursivelyClosure() with n_in_common=" << n_in_common << ", term_idx=" << term_idx << std::endl;
#endif

	term=terms[term_idx];
	/* early drop-out if \forall i in_common[i]==false is performed by the test "n_in_common>0" on the next line */
	for (std::vector<TAtom *>::const_iterator sPtr = seedTAtomsForTerm[term].begin(); n_in_common>0 && sPtr != seedTAtomsForTerm[term].end(); ++sPtr)
	{	TAtom *t;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::exploreRecursivelyClosure(): term=" << term << " considering " << (*sPtr)->name() << std::endl;
#endif

		t=*sPtr;

		assignments[term]=t->seedTAtomValue();
		is_assigned[term]=true;
		
		if (term_idx+1<terms.size())
			exploreRecursivelyClosure(tatom,term_idx+1,terms,common_value,in_common,n_in_common,value_assigned);
		else
		if (isTAtomSatisfied(tatom))
		{	for(i=0;i<terms.size();i++)
			{	if (in_common[i])
				{	if (!value_assigned[i] || common_value[i]==assignments[terms[i]])
					{	value_assigned[i]=true;
						common_value[i]=assignments[terms[i]];
					}
					else
					if (value_assigned[i] && common_value[i]!=assignments[terms[i]])
					{	in_common[i]=false;
						n_in_common--;
					}
				}
			}
#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{
std::cerr << "in_common at this stage:" << std::endl;
for(i=0;i<terms.size();i++)
	std::cerr << " " << in_common[i];
std::cerr << std::endl;
}
#endif
		}
		
		is_assigned[term]=false;
	}
}

bool ASPfSolver::closureNogoodLeftToRight(TAtom *c,bool &foundNewLits)
{	std::vector<unsigned int> allTerms,terms;
	int i;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "in ASPfSolver::closureNogoodLeftToRight(TAtom *c)" << std::endl;
#endif

	allTerms=c->getAllTerms();

	/* Extract the terms that have no assigned value */
	for (std::vector<unsigned int>::iterator j = allTerms.begin(); j != allTerms.end(); ++j)
	{	int term=*j;

		/* If a term does not have any seed t-atom, the algorithm terminates immediately.
		 * The algorithm wouldn't have any effect anyway, but terminating immediately
		 * improves efficiency.
		 */
		if (seedTAtomsForTerm[term].size()==0)
		{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "in ASPfSolver::closureNogoodLeftToRight(TAtom *c): term " << term << " has no seed t-atom. Terminating early." << std::endl;
#endif
			return(true);
		}

		if (!is_assigned[term])
		{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "term w/o assigned val: " << term << std::endl;
#endif
			terms.push_back(term);
		}
	}

	TAtomValue common_value[terms.size()];
	bool in_common[terms.size()];
	bool value_assigned[terms.size()];

	/* set in_common[i]=true and value_assigned[i]=false */
	for(i=0;i<terms.size();i++)
	{	in_common[i]=true;
		value_assigned[i]=false;
	}

	int n_in_common;
	n_in_common=terms.size();
	exploreRecursivelyClosure(c,0,terms,common_value,in_common,n_in_common,value_assigned);

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "exploreRecursivelyClosure() returned with n_in_common=" << n_in_common << std::endl;
#endif

	if (n_in_common>0)
	{	

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "bool ASPfSolver::closureNogoodLeftToRight(TAtom *c): ASPfConstraint created but I don't think the memory will ever be de-allocated!!!" << std::endl;
#endif
		ASPfConstraint *antecedents=ASPfConstraint::createStandard(this,c,true,true);

		for(i=0;i<terms.size();i++)
		{	if (in_common[i] && value_assigned[i])
			{	for (std::vector<TAtom *>::const_iterator sPtr = seedTAtomsForTerm[terms[i]].begin(); sPtr != seedTAtomsForTerm[terms[i]].end(); ++sPtr)
				{	TAtom *t;

					t=*sPtr;
#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{
if (common_value[i]==t->seedTAtomValue())
std::cerr << "bool ASPfSolver::closureNogoodLeftToRight(TAtom *c): going to force " << t->name() << std::endl;
}
#endif

					if (!s_->isTrue(t->literal()) && !s_->isFalse(t->literal()))
						foundNewLits=true;	/* only signal if the literal was not previously true or false */

					if (common_value[i]==t->seedTAtomValue() && !s_->force(t->literal(),antecedents))
						return(false);
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "  succeeded!" << std::endl;
#endif
				}
			}
				
		}
	}

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "bool ASPfSolver::closureNogoodLeftToRight(TAtom *c): returning true" << std::endl;
#endif
	return(true);
}

bool ASPfSolver::regularNogoodLeftToRight(TAtom *c,bool &foundNewLits)
{	std::vector<unsigned int> allTerms,terms;
	int i;

#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	std::cerr << "in ASPfSolver::regularNogoodLeftToRight(TAtom *c)" << std::endl;
	std::cerr << "SHOULD WE LOOP UNTIL NO MORE CHANGES ARE MADE" << std::endl;
}
#endif

	allTerms=c->getAllTerms();

	/* Extract the terms that have no assigned value */
	for (std::vector<unsigned int>::iterator j = allTerms.begin(); j != allTerms.end(); ++j)
	{	int term=*j;

		/* If a term does not have any seed t-atom, the algorithm terminates immediately.
		 * The algorithm wouldn't have any effect anyway, but terminating immediately
		 * improves efficiency.
		 */
		if (seedTAtomsForTerm[term].size()==0)
		{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "in ASPfSolver::regularNogoodLeftToRight(TAtom *c): term " << term << " has no seed t-atom. Terminating early." << std::endl;
#endif
			return(true);
		}

		if (!is_assigned[term])
		{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "term w/o assigned val: " << term << std::endl;
#endif
			terms.push_back(term);
		}
	}

	if (terms.size()>1)
	{	
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::regularNogoodLeftToRight(TAtom *c): more than one variable with unknown value -- nothing can be done." << std::endl;
#endif
		return(true);
	}

	int term=terms[0];
	bool value_found=false;
	TAtomValue the_value=std::make_pair(false,0);
	for (std::vector<TAtom *>::const_iterator sPtr = seedTAtomsForTerm[term].begin(); !value_found && sPtr != seedTAtomsForTerm[term].end(); ++sPtr)
	{	TAtom *t;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::regularNogoodLeftToRight(TAtom *c): considering " << (*sPtr)->name() << std::endl;
#endif

		t=*sPtr;

		assignments[term]=t->seedTAtomValue();
		is_assigned[term]=true;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::regularNogoodLeftToRight(TAtom *c): assigned value " << toString(t->seedTAtomValue()) << " to term " << term << std::endl;
#endif
		
		if (isTAtomSatisfied(c))
		{	value_found=true;
			the_value=t->seedTAtomValue();
		}
		
		is_assigned[term]=false;
	}

	if (!value_found)
	{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::regularNogoodLeftToRight(TAtom *c): no satisfying value found. Nothing can be done." << std::endl;
#endif
		return(true);
	}
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::regularNogoodLeftToRight(TAtom *c): satisfying value of " << toString(the_value) << " found." << std::endl;
#endif

	ASPfConstraint *antecedents=ASPfConstraint::createStandard(this,c,true,true);

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "bool ASPfSolver::regularNogoodLeftToRight(TAtom *c): ASPfConstraint created but I don't think the memory will ever be de-allocated!!!" << std::endl;
#endif

	for (std::vector<TAtom *>::const_iterator sPtr = seedTAtomsForTerm[term].begin(); sPtr != seedTAtomsForTerm[term].end(); ++sPtr)
	{	TAtom *t;

		t=*sPtr;
#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{
if (the_value==t->seedTAtomValue())
std::cerr << "bool ASPfSolver::regularNogoodLeftToRight(TAtom *c): going to force ~" << t->name() << std::endl;
}
#endif

		if (!s_->isTrue(t->literal()) && !s_->isFalse(t->literal()))
			foundNewLits=true;	/* only signal if the literal was not previously true or false */

		if (the_value==t->seedTAtomValue() && !s_->force(~t->literal(),antecedents))
			return(false);
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "  succeeded!" << std::endl;
#endif
	}

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "bool ASPfSolver::regularNogoodLeftToRight(TAtom *c): returning true" << std::endl;
#endif
	return(true);
}

// Closure nogoods:
// { \T f = g, \F f=1, \F f=2, \F f=3, \F f=4, ... }
// { \T f = g, \F g=1, \F g=2, \F g=3, \F g=4, ... }
// ...
// if all of { \F f=1, \F f=2, \F f=3, \F f=4, ... }
// are true (i.e. f=v is false for every v),
// then \T f=g cannot be true (i.e. f=g must be false).
//
bool ASPfSolver::closureNogoodTriggered(TAtom *c)
{	std::vector<unsigned int> terms;

	terms=c->getAllTerms();
	
	for (std::vector<unsigned int>::iterator j = terms.begin(); j != terms.end(); ++j)
	{	int term=*j;

#ifdef ASPF_DEBUG_LIGHT
int num_tested=0;
#endif
		bool allFalse=true;
#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	if (seedTAtomsForTerm[term].size()==0)
	std::cerr << "OOPS: no seed t-atoms found for term " << term << ". Special nogood will be triggered..." << std::endl;
}
#endif
		for (std::vector<TAtom *>::const_iterator sPtr = seedTAtomsForTerm[term].begin(); allFalse && sPtr != seedTAtomsForTerm[term].end(); ++sPtr)
		{	TAtom *s;

			s=*sPtr;
#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	num_tested++;
	std::cerr << "t-atom " << s->name() << "... ";
}
#endif

			if (!s_->isFalse(s->literal()))
			{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "NOT false!" << std::endl;
#endif
				allFalse=false;
			}
#ifdef ASPF_DEBUG_LIGHT
else
if (debug_) std::cerr << "false!" << std::endl;
#endif
		}
		if (allFalse) return(true);	// triggered!
	}

	return(false);
}

bool ASPfSolver::recordTermAssignment(TAtom *c)
{
#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{
TermExpression *c1,*c2;
c->getRelation(c1,c2);
std::cerr << "got variable: "; displayTermExpression(c1); std::cerr << "; int="; displayTermExpression(c2); std::cerr << std::endl;
}
#endif

	if (is_assigned[c->seedTAtomTermIndex()] && assignments[c->seedTAtomTermIndex()]!=c->seedTAtomValue())
	{	
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "aspfsolver:recordTermAssignment() detected inconsistency on the FACTS: " << toString(assignments[c->seedTAtomTermIndex()]) << " vs " << toString(c->seedTAtomValue()) << std::endl;
#endif
		return(false);
	}
	if (!is_assigned[c->seedTAtomTermIndex()])
	{	
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "recording assignment: term " << c->seedTAtomTermIndex() << " set to value " << toString(c->seedTAtomValue()) << std::endl;
#endif
		assignments[c->seedTAtomTermIndex()]=c->seedTAtomValue();
		is_assigned[c->seedTAtomTermIndex()]=true;
		assignment_level[c->seedTAtomTermIndex()]=s_->decisionLevel();
	}
#ifdef ASPF_DEBUG_LIGHT
	else
if (debug_) std::cerr << "assignment already recorded: term " << c->seedTAtomTermIndex() << ", value " << toString(c->seedTAtomValue()) << std::endl;
#endif
	
	return(true);
}


bool ASPfSolver::bruteForcePropagate(/*LitVec &lv*/bool &foundNewLits)
{/* [marcy 011712] check if the current assignments cause any t-atoms to be true or false */

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "in ASPfSolver::bruteForcePropagate()" << std::endl;
#endif

	foundNewLits=false;
	for (std::map<Var, std::vector<TAtom *> >::iterator i = tatoms_.begin(); i != tatoms_.end(); ++i)
	{	
		for (std::vector<TAtom *>::iterator tatomIter = (i->second).begin(); tatomIter != (i->second).end(); ++tatomIter)
		{	TAtom *c;
			TermExpression *c1,*c2;
			int rel;
			TAtomValue v1,v2;
			bool satisfied;
			bool reasonTruthValue;

			c=*tatomIter;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "CLASP_VAR " << c->claspVar() << " has a name " << c->name() << std::endl;
#endif

			if (!c->getType()==CSPConstraint::RELATION) continue;

#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	std::cerr << "checking if constraint affected: ";
	displayTermExpression(c->cspconstraintPtr()); 
	std::cerr << std::endl;
}
#endif
		    	rel=c->getRelation(c1,c2);
			satisfied=false;
			reasonTruthValue=true;

			if (!getValueOfTermExpression(c1,v1) || !getValueOfTermExpression(c2,v2))
			{	
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "  not all terms have a value!" << std::endl;
#endif

				/* We can do the check here because the if statement above
				 * guarantees that some variables don't have a value yet.
				 */
				if (c->isSeedTAtom()) continue;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "found dependent t-atom " << c->name() << std::endl;
#endif

#ifdef USE_LEFT_TO_RIGHT_NOGOODS
				if (s_->isTrue(c->literal()) && !closureNogoodLeftToRight(c,foundNewLits))
					return(false);
				else
				if (s_->isFalse(c->literal()) && !regularNogoodLeftToRight(c,foundNewLits))
					return(false);
				// otherwise, keep checking to see if the closureNogood or regular nogood are triggered
#else
				std::cerr << "WARNING!!! Left-to-right nogoods DISABLED in aspfsolver.cpp:bruteForcePropagate()!!!" << std::endl;
#endif

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "closureNogood check on constraint " << c->name() << std::endl;
#endif
				if (closureNogoodTriggered(c))
				{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "  specialNogood TRIGGERED!!!" << std::endl;
#endif
					/* just do nothing -- leave satisfied = false and let the literal be added below */
					reasonTruthValue=false;
				}
				else
				{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "  specialNogood not triggered" << std::endl;
#endif
					continue;
				}
			}
			else
				satisfied=isSatisfied(v1,rel,v2);

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "  making t-atom " << c->name() << ((satisfied) ? " true":" false") << std::endl;
#endif

			if (!s_->isTrue(c->literal()) && !s_->isFalse(c->literal()))
				foundNewLits=true;	/* only signal if c->literal() was not already true or false */

			if (!s_->force((satisfied ? c->literal():~(c->literal())), ASPfConstraint::createStandard(this,c,false,reasonTruthValue)))
				return(false);
		}
	}
	return(true);
}
/* */

bool ASPfSolver::initialize(ProgramBuilder* b)
{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "in ASPfSolver::initialize()" << std::endl;
#endif
	// convert uids to solver literal ids
// currently no preprocessing support for constants, so this will be disabled
//	if (!addedDomain_)
//		throw ASPmCSPException("You must have at least one \"$domain(l..u).\" statement in the program");
/* [marcy 012612] */

#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	/* show the assignment of variables to atoms */
	//for (AtomIndex::size_type i = 0; i != index->size(); ++i) {
	std::cerr << "atom/variable mapping" << std::endl;
	for(int idx=0;idx<b->stats.index.size();idx++)
	{	std::cerr << idx << ") atom=" << b->stats.index[idx].name << " var=" << b->stats.index[idx].lit.var() << "; trueValue=" << (int)trueValue(b->stats.index[idx].lit) << "; sign= " << (int)b->stats.index[idx].lit.sign() << std::endl;
	}
	std::cerr << "---------------------" << std::endl;
}
#endif

/* set marcyExtraAtom to false, with no reason given */
if (b->extraAtomId!=-1)
{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) 	std::cerr << "attempting to set the extra literal to false" << std::endl;
#endif
	if (!s_->force(~(b->stats.index[b->extraAtomId].lit), 0))
		return(false);
#ifdef ASPF_DEBUG_LIGHT
if (debug_) 
{	std::cerr << "var " << b->stats.index[b->extraAtomId].lit.var() << " is atom " << b->stats.index[b->extraAtomId].name << std::endl;
	std::cerr << "     isTrue=" << s_->isTrue(b->stats.index[b->extraAtomId].lit) << "; isFalse=" << s_->isFalse(b->stats.index[b->extraAtomId].lit) << std::endl;
}
#endif
}
/* */
	// convert constraints to literal variables
	std::map<Var, std::vector<TAtom *> > newConstraints;
	for (std::map<Var, std::vector<TAtom *> >::iterator i = tatoms_.begin(); i != tatoms_.end(); ++i)
	{

/* [marcy 011712] */
		/* By construction of tatoms_ up to this point,
		 * it is guaranteed that there will be a single element
		 * in the std::vector of each key.
		 */

		TAtom *c=i->second[0];

		newConstraints[b->stats.index[i->first].lit.var()].push_back(new TAtom(*c));

		delete c;

		/* point c to the newly created object in newConstraints */
		c=newConstraints[b->stats.index[i->first].lit.var()].back();

		c->setLiteral(b->stats.index[i->first].lit);
		c->setClaspVar(b->stats.index[i->first].lit.var());
		c->setTrueValue(trueValue(b->stats.index[i->first].lit));
		c->setName(b->stats.index[i->first].name);

#ifdef ASPF_DEBUG_LIGHT
if (debug_) 
{	std::cerr << "clasp var " << c->claspVar() << " / uid " << i->first << " / index " << c->literal().index() << " is the t-atom " << c->name() << " i.e.: ";
	displayTermExpression(c->cspconstraintPtr());
	std::cerr << "     isTrue=" << s_->isTrue(c->literal()) << "; isFalse=" << s_->isFalse(c->literal()) << std::endl;
	std::cerr << "     has name: " << c->name() << std::endl;
}
#endif

if (c->isSeedTAtom()) seedTAtomsForTerm[c->seedTAtomTermIndex()].push_back(c);

if (s_->isTrue(b->stats.index[i->first].lit) && c->isSeedTAtom())
{	if (!recordTermAssignment(c))
		return(false);
}
/* */

	}

	tatoms_ = newConstraints;

/* [marcy 011912] */
{
/* Early propagation, for the literals made true by endProgram(), to which regular propagate() is not applied by clasp  */
	//LitVec lv;

	bool foundNewLits;
	if (!bruteForcePropagate(/*lv*/foundNewLits))
		return(false);	// conflict!

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "no longer using propagateNewLiteralsToClasp()!!" << std::endl;
#endif
	//if (!propagateNewLiteralsToClasp(lv))
	//	return(false);	// conflict!

	return(true);
}
/* */
}

bool ASPfSolver::hasAnswer()
{
/* [marcy 020812] */
	return(true);
/* */
}

bool ASPfSolver::nextAnswer()
{
/* [marcy 020812] */
	return(false);
/* */
}

void ASPfSolver::printAnswer()
{
/* [marcy 020812] */
{
	for (int i=0;i<n_terms;i++)
	{	bool forceShown=false,forceHidden=false;

		if (seedTAtomsForTerm[i].size()>0)
		{	forceShown=seedTAtomsForTerm[i][0]->isShown();
			forceHidden=seedTAtomsForTerm[i][0]->isHidden();
		}
		
		if (is_assigned[i] && (!hideAll_ || forceShown) && (!forceHidden || forceShown))
		{	std::cout << variables_[i] << "#=";
			if (assignments[i].first)
				std::cout << variables_[assignments[i].second];
			else
				std::cout << assignments[i].second;
			std::cout << " ";
		}
	}

}
/* */
}

bool ASPfSolver::isConstraintLiteral(Literal l)
{
	//TODO: this urgendtly needs to be changed, this is called quite often
	return tatoms_.find(l.var()) != tatoms_.end();
}

bool ASPfSolver::propagate(const LitVec& lv, bool& foundNewLits)
{
/* [marcy 011712] */
	if (lv.size()>0)
	{
/* [marcy 011712] */
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "propagate() --->" << std::endl;
#endif
		for (LitVec::const_iterator i = lv.begin(); i != lv.end(); ++i)
		{	for (std::vector<TAtom *>::iterator tatomIter = tatoms_[i->var()].begin(); tatomIter != tatoms_[i->var()].end(); ++tatomIter)
			{	TAtom *t;

				t=*tatomIter;


#ifdef ASPF_DEBUG_LIGHT
if (debug_) 
{	std::cerr << "PROPAGATE: ";
	displayTermExpression(t->cspconstraintPtr());
	std::cerr << "     isTrue=" << s_->isTrue(t->literal()) << "; isFalse=" << s_->isFalse(t->literal()) << std::endl;
}
#endif

				if (t->isSeedTAtom() && s_->isTrue(t->literal()))
				{	if (!recordTermAssignment(t))
					{	/* use force() to let clasp handle the setup for the conflict.
						 * force() at this point is guaranteed to return false.
						 */
						s_->force(~t->literal(), ASPfConstraint::createStandard(this,t,false,true));
						return(false);
					}
				}
			}
		}

		foundNewLits = false;

		//LitVec newlv;
		if (!bruteForcePropagate(/*newlv*/foundNewLits))
			return(false);	// conflict!

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "no longer using propagateNewLiteralsToClasp()!!!" << std::endl;
#endif
/*
		if (newlv.size()>0)
		{	// propagate new literals to clasp
			if (!propagateNewLiteralsToClasp(newlv))
			{
#ifdef DEBUGTEXT
				std::cerr << "conflict detected while propagating to clasp" << std::endl;
#endif
				return(false);
			}
		}
		foundNewLits = true;
*/
	}

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "------------------" << std::endl;
#endif
	return(true);
/* */
}


void ASPfSolver::undoUntil(unsigned int level)
{
/* [marcy 020712] */
	int i;

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "Backjump to level(+1?) " << level << " from level solver: " << s_->decisionLevel() << std::endl;
#endif

	for(i=0;i<n_terms;i++)
	{	if (is_assigned[i] && assignment_level[i]>level)
		{
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "undoing assignment to term #" << i << "; val=" << toString(assignments[i]) << "; old_level=" << assignment_level[i] << std::endl;
#endif
			is_assigned[i]=false;
		}
	}
/* */
}


/*
bool ASPfSolver::propagateNewLiteralsToClasp(const LitVec& newLits)
{
#ifdef ASPF_DEBUG_LIGHT
	std::cerr << "before propagateToClasp: " << std::endl;
#endif

	for (LitVec::const_iterator i = newLits.begin(); i != newLits.end(); ++i)
	{
#ifdef ASPF_DEBUG_LIGHT
std::cerr << "propagating " << claspVarToNames[i->var()][0] << std::endl;
#endif
		if (!s_->force(*i, &aspfReason_))
		{	
#ifdef ASPF_DEBUG_LIGHT
std::cerr << "conflict!!" << std::endl;
#endif
			return false;
		}
	}
	return true;
}
*/

bool ASPfSolver::isSeedTAtom(CSPConstraint *c)
{	TAtom t(*c);
	return(t.isSeedTAtom());
}

ASPfSolver::TAtom::TAtom() : cspconstraint_(), isSeedTAtom_(false), shown_(false), hidden_(false)
{
}

ASPfSolver::TAtom::TAtom(CSPConstraint c) : cspconstraint_(c), shown_(false), hidden_(false)
{
	is_constant_[0]=is_constant_[1]=false;

	detectSeedTAtom();
}

/* [marcy 011712] */
void ASPfSolver::TAtom::detectSeedTAtom()
{	TermExpression *c1,*c2;
	bool isEQ;

#ifdef ASPF_DEBUG_LIGHT
if (debug_)
{	if (getType()==CSPConstraint::RELATION &&
	    getRelation(c1,c2)==CSPConstraint::EQ)
	{
	std::cerr << "c1.type=" << c1->getType() << " "
		  << "c2.type=" << c2->getType() << std::endl;
	}
}
#endif

	isEQ=(getType()==CSPConstraint::RELATION &&
	      getRelation(c1,c2)==CSPConstraint::EQ);

	is_constant_[0]=!c1->isFTerm();
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::TAtom::detectSeedTAtom(): LHS of " << name() << " is " << (is_constant_[0] ? "constant":"f-term") << std::endl;
#endif
	is_constant_[1]=!c2->isFTerm();
#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::TAtom::detectSeedTAtom(): RHS of " << name() << " is " << (is_constant_[1] ? "constant":"f-term") << std::endl;
#endif
	

	isSeedTAtom_=(isEQ && 
		     ((!is_constant_[0] && is_constant_[1]) ||
		      (is_constant_[0] && !is_constant_[1])));

	if (isSeedTAtom())
	{	if (!is_constant_[0])
		{	seedTAtomTermIndex_=c1->getVar();
			shown_=c1->isShown();
			hidden_=c1->isHidden();
/* [marcy 041112] */
			if (c2->getType()==CSPConstraint::VARIABLE)
				seedTAtomValue_=std::make_pair(true,c2->getVar());
			else
				//seedTAtomValue_=c2->getInteger();
				seedTAtomValue_=std::make_pair(false,c2->getInteger());
/* */
		}
		else
		{	seedTAtomTermIndex_=c2->getVar();
			shown_=c2->isShown();
			hidden_=c2->isHidden();
/* [marcy 041112] */
			if (c1->getType()==CSPConstraint::VARIABLE)
				seedTAtomValue_=std::make_pair(true,c1->getVar());
			else
				//seedTAtomValue_=c2->getInteger();
				seedTAtomValue_=std::make_pair(false,c1->getInteger());
/* */
		}

#ifdef ASPF_DEBUG_LIGHT
if (debug_) std::cerr << "ASPfSolver::TAtom::detectSeedTAtom(): seed t-atom detected; fterm=" << seedTAtomTermIndex_ << "; val=" << toString(seedTAtomValue_) << std::endl;
#endif
	}
}

std::vector<unsigned int> ASPfSolver::TAtom::getAllTerms(TermExpression *c)
{	std::vector<unsigned int> r;
	TermExpression *c1,*c2;

	c1=NULL;
	c2=NULL;
	switch(c->getType())
	{	case CSPConstraint::INTEGER:
			break;
		case CSPConstraint::VARIABLE:
			if (c->isFTerm())
				r.push_back(c->getVar());
			break;
		case CSPConstraint::OPERATOR:
			c->getRelation(c1,c2);
			break;
		case CSPConstraint::RELATION:
			c->getOperator(c1,c2);
			break;
		default:
			std::cerr << "internal error: unknown term type " << c->getType() << std::endl;
			exit(1);
	}
	if (c1)
	{	std::vector<unsigned int> v(getAllTerms(c1));
		r.insert(r.end(), v.begin(), v.end());
	}
	if (c2)
	{	std::vector<unsigned int> v(getAllTerms(c2));
		r.insert(r.end(), v.begin(), v.end());
	}

	return(r);

}

}//namespace

